/**
 * Handles UI-specific interactions for the customer journey.
 * @param {Object} cjui - this module
 * @param {Object} cj - the customerJourney.js module
 * @param {Object} $j - the jQuery libary. Should include jQuery UI or a similar widget library.
 */
(function (cjui, cj, global, $j) {
    "use strict";

    var helpDialogId = '#contextDisplayDialog';

    var subscribePanelId = '#subscribePanel';
    var subscribeEmailId = '#subscribeEmail';
    var subscribeEntryId = '#subscribeInputDiv';
    var subscribeControlsId = '#subscribeControls';
    var subscribeResultsId = '#subscribeStatus';

    function createHelpDialog() {
        var dialog = $j(helpDialogId).length;
        if (dialog === 0) {
            console.info('CustomerJourney: Help dialog does not exist');
            dialog = document.createElement('div');
            dialog.id = helpDialogId.substring(1);
            
            // display the context here
            var contextDisplay = document.createElement("pre");
            dialog.appendChild(contextDisplay);
            
            // button to manually update the schema
            var updateContextBtn = document.createElement("button");
            updateContextBtn.onclick = cj.updateSchema;
            updateContextBtn.textContent = "Update schema";
            updateContextBtn.className = "btn btn-default";
            dialog.appendChild(updateContextBtn);
            
            // button to manually display the new schema
            var displayContextBtn = document.createElement("button");
            displayContextBtn.onclick = cjui.showContext;
            displayContextBtn.textContent = "Show current schema";
            displayContextBtn.className = "btn btn-default";
            dialog.appendChild(displayContextBtn);
            
            document.body.appendChild(dialog);
        }
    }

    // creates a dialog to allow the user to subscribe/request a customer ID
    // this will be positioned 150 pixels below the subscribe link
    // return false if we already have a context
    function createSubscribeDialog() {
        if (sessionStorage.getItem("contextId")) {
            console.info("Context already exists - we don't need a customer ID");
            return false;
        }
        
        $j(subscribePanelId).dialog({
            width: '300px',
            height: 'auto',
            create: function () {
                $j(this).parent().css({
                    'position': 'fixed'
                });
            },
            position: {
                my: 'top left',
                at: 'top+150',
                of: $j('#subscribeLink'),
                collision: 'fit'
            }
        });
        return true;
    }

    function hideSubscribeDialog() {
        $j(subscribePanelId).dialog('close');
    }

    /**
     * Show the full context object for debugging.
     */
    cjui.showContext = function () {
        var context = cj.getContext();
        $j(helpDialogId).dialog().children("pre").text(JSON.stringify(context));
    };

    /**
     * Sample method to subscribe via email
     */
    cjui.subscribe = function () {
        var email = $j(subscribeEmailId).val();
        if (global.isStringEmpty(email)) {
            $j('#subscribeStatus').text('Please enter your email address!');
        } else {
            cj.requestCustomerId('email', email);
        }
    };

    cjui.hideSubscribeDialog = function () {
        $j(subscribePanelId).dialog('close');
    };

    cjui.showEmailDiv = function () {
        $j(subscribeEntryId).show();
        $j(subscribeControlsId).hide();
    };

    cjui.setSubscribeStatus = function (statusText) {
        $j(subscribeResultsId).text(statusText);
    };

    /**
     * Close the subscribe panel in two seconds.
     */
    cjui.closeSubscribePanel = function () {
        $j(subscribeEntryId).children().hide();
        $j(subscribeResultsId).show();
        setTimeout(hideSubscribeDialog, 2000);
    };

    cjui.setup = function () {
        $(".button").button();
        customerJourney.setUI(cjui);
        createHelpDialog();
        if (!createSubscribeDialog()) {
            customerJourney.setup();
        }
        
    };

    return cjui;
})(window.customerJourneyUI = window.customerJourneyUI || {}, window.customerJourney, window.avayaGlobal, window.jQuery);